<?php
/**
 * Blesta Beta Order Type
 *
 * @package blesta
 * @subpackage blesta.plugins.order.lib.order_types
 * @copyright Copyright (c) 2010, Phillips Data, Inc.
 * @license http://www.blesta.com/license/ The Blesta License Agreement
 * @link http://www.blesta.com/ Blesta
 */
class OrderTypeBlestaBeta extends OrderType
{
    /**
     * @var string The URL to make the license check request to
     */
    private static $legacy_license_url = 'https://phillipsdata.com/ams/license_verify.php';

    /**
     * @var string The authors of this order type
     */
    private static $authors = [['name'=>'Phillips Data, Inc.','url'=>'http://www.blesta.com']];

    /**
     * Construct
     */
    public function __construct()
    {
        Language::loadLang('order_type_blesta_beta', null, dirname(__FILE__) . DS . 'language' . DS);

        Loader::loadComponents($this, ['Input']);
    }

    /**
     * Sets the SessionCart being used by the order form
     *
     * @param SessionCart $cart The session cart being used by the order form
     */
    public function setCart(SessionCart $cart)
    {
        parent::setCart($cart);

        $this->cart->setCallback('addItem', [$this, 'verifyAddItem']);
    }

    /**
     * Returns the name of this order type
     *
     * @return string The common name of this order type
     */
    public function getName()
    {
        return Language::_('OrderTypeBlestaBeta.name', true);
    }

    /**
     * Returns the name and URL for the authors of this order type
     *
     * @return array The name and URL of the authors of this order type
     */
    public function getAuthors()
    {
        return self::$authors;
    }

    /**
     * Determines whether or not the order type supports multiple package groups or just a single package group
     *
     * @return mixed If true will allow multiple package groups to be selected,
     *  false allows just a single package group, null will not allow package selection
     */
    public function supportsMultipleGroups()
    {
        return false;
    }

    /**
     * Determines whether or not the order type requires the perConfig step of
     * the order process to be invoked.
     *
     * @return bool If true will invoke the preConfig step before selecting a package,
     *  false to continue to the next step
     */
    public function requiresPreConfig()
    {
        return true;
    }

    /**
     * Handle an HTTP request. This allows an order template to execute custom code
     * for the order type being used, allowing tighter integration between the order type and the template.
     * This can be useful for supporting AJAX requests and the like.
     *
     * @param array $get All GET request parameters
     * @param array $post All POST request parameters
     * @param array $files All FILES request parameters
     * @return string HTML content to render (if any)
     */
    public function handleRequest(array $get = null, array $post = null, array $files = null)
    {
        $this->view = new View();
        $this->view->setDefaultView('plugins' . DS . 'order' . DS);
        $this->view->setView(
            null,
            'templates' . DS . $this->order_form->template . DS . 'types' . DS . $this->order_form->type
        );

        // Load the helpers required for this view
        Loader::loadHelpers(
            $this,
            ['Form', 'Html', 'WidgetClient', 'CurrencyFormat' => [Configure::get('Blesta.company_id')]]
        );

        $alert = new View();
        $alert->set('notice', Language::_('OrderTypeBlestaBeta.!notice.license.logged', true));
        $alert->Html = $this->Html;
        $this->view->set('message', $alert->fetch('message'));

        if (!empty($post)) {
            $post['license_key'] = trim($post['license_key']);

            if ($this->validateLegacyKey($post['license_key'])) {
                // Log key as being used
                $this->logKey($post['license_key']);

                // Flag that the license is valid
                $this->cart->setData('license_valid', true);

                $package_group_id = $this->order_form->groups[0]->package_group_id;
                $package = $this->getPackage($package_group_id);

                // If license is valid, redirect to add the package
                header(
                    'Location: ' . $get['base_uri'] . 'plugin/order/main/configure/'
                    . $this->order_form->label . '/?pricing_id=' . $package->pricing[0]->id
                    . '&group_id=' . $package_group_id
                );
                exit;
            }
        }

        $this->view->set('base_uri', $get['base_uri']);
        $this->view->set('order_form', $this->order_form);
        $this->view->set('vars', (object)$post);

        return $this->view->fetch('lookup');
    }

    /**
     * Verifies that the item can be added, empties the cart on failure
     *
     * @param EventInterface $event The event
     */
    public function verifyAddItem(EventInterface $event)
    {
        if (!$this->cart->getData('license_valid')) {
            $this->cart->emptyCart();
        }

        // Remove the flag
        $this->cart->setData('license_valid', false);
    }

    /**
     * Fetches the first package from the given package group
     *
     * @param int $package_group_id The ID of the package group to fetch a package for
     * @return mixed A stdClass object representing the package, false if no package exists
     */
    private function getPackage($package_group_id)
    {
        Loader::loadModels($this, ['Packages']);

        $packages = $this->Packages->getAllPackagesByGroup($package_group_id, null, ['hidden' => true]);
        if (isset($packages[0])) {
            return $packages[0];
        }
        return false;
    }

    /**
     * Validate a legacy Blesta license key
     *
     * @param string $license_key A 16-character license key
     * @return bool True if valid, false otherwise. Sets Input errors on error
     */
    private function validateLegacyKey($license_key)
    {
        Loader::loadComponents($this, ['Net']);
        $this->Http = $this->Net->create('Http');
        $url = self::$legacy_license_url;

        if ($this->keyInLog($license_key)) {
            return false;
        }

        try {
            $response = json_decode($this->Http->post($url, ['license' => $license_key]));
        } catch (Exception $e) {
            // Server is down/unresponsive
            $this->Input->setErrors(
                [
                    'license' => [
                        'bad_response' => Language::_('OrderTypeBlestaBeta.!errors.license.bad_response', true)
                    ]
                ]
            );
            return false;
        }

        if (!$response) {
            $this->Input->setErrors(
                [
                    'license' => [
                        'bad_response' => Language::_('OrderTypeBlestaBeta.!errors.license.bad_response', true)
                    ]
                ]
            );
            return false;
        }

        if ($response->valid) {
            return true;
        }

        $this->Input->setErrors(
            ['license' => ['invalid' => Language::_('OrderTypeBlestaBeta.!errors.license.invalid', true)]]
        );
        return false;
    }

    /**
     * Checks if the license key exists in the log
     *
     * @param string $license_key The license key to check for
     * @return bool True if the key exists in the log, false otherwise. Sets Input errors if found
     */
    private function keyInLog($license_key)
    {
        $license_key = trim($license_key);
        $fh = fopen(dirname(__FILE__) . DS . 'log.txt', 'r');
        while ($license_key != '' && ($buffer = fgets($fh)) !== false) {
            if (strpos($buffer, $license_key) !== false) {
                $this->Input->setErrors(
                    ['license' => ['logged' => Language::_('OrderTypeBlestaBeta.!errors.license.logged', true)]]
                );
                return true;
            }
        }
        fclose($fh);

        return false;
    }

    /**
     * Records the license key's useage to the log
     *
     * @param string $license_key The license ke to log
     */
    private function logKey($license_key)
    {
        $requestor = $this->getFromContainer('requestor');

        $entry = date('c') . "\t" . $requestor->ip_address . "\t" . $license_key . "\n";

        file_put_contents(dirname(__FILE__) . DS . 'log.txt', $entry, FILE_APPEND);
    }
}
